<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform" xmlns:fo="http://www.w3.org/1999/XSL/Format">
	
	
	<!-- Standard decimal format for swiss numbers. Empty numbers are shown as empty. -->
	<xsl:decimal-format grouping-separator="'" decimal-separator="." NaN=""/>
	<!-- Format string to use for integers -->
	<xsl:variable name="integer-format">###'###'###'###</xsl:variable>
	<!-- Format string to use for amounts including cents -->
	<xsl:variable name="cents-format">###'###'###'##0.00</xsl:variable>
	<!-- Format string to use for percentages. % sign is not added -->
	<xsl:variable name="percent-format">###.##</xsl:variable>
	
	<!-- Page margins -->
	<xsl:variable name="page-margins">13mm 13mm 8mm 13mm</xsl:variable>
	<xsl:variable name="first-header-height">0mm</xsl:variable>
	<xsl:variable name="header-height">11mm</xsl:variable>
	<xsl:variable name="footer-height">8mm</xsl:variable>
	<xsl:variable name="first-body-top-margin">0mm</xsl:variable>
	<xsl:variable name="body-top-margin">0mm</xsl:variable>
	<xsl:variable name="body-top-margin-test">11mm</xsl:variable>
	<xsl:variable name="body-bottom-margin">12mm</xsl:variable>
	
	<!-- Some standard definitions for the layout of data blocks in PDF message layout -->
	<!-- White space between data blocks -->
	<xsl:variable name="space-after-blocks">6mm</xsl:variable>
	<!-- Font to use -->
	<xsl:variable name="font-family">Liberation Sans</xsl:variable>
	
	<!-- Width of the left header column on first page -->
	<xsl:variable name="header-left-column-width">92mm</xsl:variable>
	<!-- Width of the right header column on first page -->
	<xsl:variable name="header-right-column-width">proportional-column-width(1)</xsl:variable>
	<!-- Width of the title column (leftmost column of a data block) -->
	<xsl:variable name="title-column-width">30mm</xsl:variable>
	<!-- Width of the description column (second column of a data block) -->
	<xsl:variable name="description-column-width">40mm</xsl:variable>
	<!-- Width of the description column (second column of a data block), when in a table -->
	<xsl:variable name="description-column-table-width">proportional-column-width(1)</xsl:variable>
	<!-- Width of an interest column (showing interest rate) -->
	<xsl:variable name="interest-column-width">13mm</xsl:variable>
	<!-- Width of value column (showing amounts) -->
	<xsl:variable name="value-column-width">25mm</xsl:variable>
	<!-- Width of doctype column (showing amounts) -->
	<xsl:variable name="docpath-column-width">50mm</xsl:variable>
	<!-- Width of doctype column (showing amounts) -->
	<xsl:variable name="doctype-column-width">26mm</xsl:variable>
	
	<!-- Colordefinition for grey background -->
	<xsl:variable name="grey-background">#cdcdcd</xsl:variable>
	
	<xsl:variable name="lower-case">abcdefghijklmnopqrstuvwxyzäöüàéè</xsl:variable>
	<xsl:variable name="upper-case">ABCDEFGHIJKLMNOPQRSTUVWXYZÄÖÜÀÉÈ</xsl:variable>
	
	<!-- ************* Formatting templates *************** -->
	
	<!--
		Format a date for display in PDF.
		@param value the date to be formatted (as defined in xs:date)
	-->
	<xsl:template name="format-date">
		<xsl:param name="value"/>
		<xsl:param name="yearDigits" select="4"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="string-length($trimmed-value) > 0">
				<xsl:variable name="year" select="substring($trimmed-value, 5 - $yearDigits, $yearDigits)"/>
				<xsl:variable name="month" select="substring($trimmed-value, 6, 2)"/>
				<xsl:variable name="day" select="substring($trimmed-value, 9, 2)"/>
				<xsl:value-of select="$day"/>.<xsl:value-of select="$month"/>.<xsl:value-of select="$year"/>
			</xsl:when>
		</xsl:choose>
	</xsl:template>
	
		<!--
		Format a date with time for display in PDF.
		@param value the date to be formatted (as defined in xs:date)
	-->
	<xsl:template name="format-datetime">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="string-length($trimmed-value) > 0">
				<xsl:variable name="year" select="substring($trimmed-value, 1, 4)"/>
				<xsl:variable name="month" select="substring($trimmed-value, 6, 2)"/>
				<xsl:variable name="day" select="substring($trimmed-value, 9, 2)"/>
				<xsl:variable name="hour" select="substring($trimmed-value, 12, 2)"/>
				<xsl:variable name="min" select="substring($trimmed-value, 15, 2)"/>
				<xsl:variable name="sec" select="substring($trimmed-value, 18, 2)"/>
				<xsl:value-of select="$day"/>.<xsl:value-of select="$month"/>.<xsl:value-of select="$year"/> / <xsl:value-of select="$hour"/>:<xsl:value-of select="$min"/>:<xsl:value-of select="$sec"/>
			</xsl:when>
		</xsl:choose>
	</xsl:template>
	<!--
		Format a partially known date for display in PDF.
		@param value the date to be formatted (as defined in xs:data, xs:gYearMonth, xs:gYear)
	-->
	<xsl:template name="format-datePartiallyKnownType">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="string-length($trimmed-value) > 6">
				<xsl:call-template name="format-date">
					<xsl:with-param name="value" select="$value"/>
				</xsl:call-template>
			</xsl:when>
			<xsl:when test="string-length($trimmed-value) > 4">
				<xsl:variable name="year" select="substring($value, 1, 4)"/>
				<xsl:variable name="month" select="substring($value, 6, 2)"/>
				<xsl:value-of select="$month"/>.<xsl:value-of select="$year"/>
			</xsl:when>
			<xsl:when test="string-length($trimmed-value) > 0">
				<xsl:value-of select="$value"/>
			</xsl:when>
		</xsl:choose>
	</xsl:template>
	
	<!--
		Format a date with month and year for display in PDF.
		@param value the date to be formatted (YYYY-MM)
	-->
	<xsl:template name="format-month">
		<xsl:param name="value"/>
		<xsl:param name="delimiter" select="'/'"/>
		<xsl:param name="yearDigits" select="2"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="string-length($trimmed-value) > 0">
				<xsl:variable name="year" select="substring($trimmed-value, 5 - $yearDigits, $yearDigits)"/>
				<xsl:variable name="month" select="substring($trimmed-value, 6, 2)"/>
				<xsl:value-of select="$month"/><xsl:value-of select="$delimiter"/><xsl:value-of select="$year"/>
			</xsl:when>
		</xsl:choose>
	</xsl:template>	
	
	<!--
		Format a vn (=new AHV number), i.e. insert dots where appropriate.
		@param value vn to format
	-->
	<xsl:template name="format-vnType">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="string-length($trimmed-value) = 13">
				<xsl:value-of select="substring($trimmed-value, 1, 3)"/>.<xsl:value-of select="substring($trimmed-value, 4, 4)"/>.<xsl:value-of select="substring($trimmed-value, 8, 4)"/>.<xsl:value-of select="substring($trimmed-value, 12, 2)"/>
			</xsl:when>
			<xsl:otherwise>
				<xsl:value-of select="$trimmed-value"/>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	<!--
		Format an old AHV number, i.e. insert dots where appropriate.
		@param value vn to format
	-->
	<xsl:template name="format-old-ahv">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="string-length($trimmed-value) = 11">
				<xsl:value-of select="substring($trimmed-value, 1, 3)"/>.<xsl:value-of select="substring($trimmed-value, 4, 2)"/>.<xsl:value-of select="substring($trimmed-value, 6, 3)"/>.<xsl:value-of select="substring($trimmed-value, 9, 3)"/>
			</xsl:when>
			<xsl:otherwise>
				<xsl:value-of select="$trimmed-value"/>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	<!--
		Format a string, i.e. output the string
		@param value string to format
	-->
	<xsl:template name="format-string">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:value-of select="$trimmed-value"/>
	</xsl:template>
	
	<!--
		Format a string as a title, i.e. make sure the first character is upper case
		@param value string to format
	-->
	<xsl:template name="format-title">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:variable name="first-char" select="translate(substring($trimmed-value, 1, 1), $lower-case, $upper-case)"/>
		<xsl:value-of select="concat($first-char, substring($trimmed-value, 2))"/>
	</xsl:template>
	
	<!--
		Format a percentage, format a number and append a percent sign. If empty, the empty sign
		is output instead.
		@param value string to format
		@param empty-sign string to output if value is empty
	-->
	<xsl:template name="format-percent">
		<xsl:param name="value"/>
		<xsl:param name="empty-sign"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="string-length($trimmed-value) > 0">
				<xsl:value-of select="format-number($trimmed-value, $percent-format)"/>%
			</xsl:when>
			<xsl:otherwise><xsl:value-of select="$empty-sign"/></xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	
	<!-- ************* Decodings templates *************** -->
	
	
	<!--
		Decode a MIME type to a human readable string.
		@param value the type to be decoded
	-->
	<xsl:template name="decode-mime">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="$trimmed-value = 'image/tiff'">TIFF</xsl:when>
			<xsl:when test="$trimmed-value = 'application/pdf'">PDF</xsl:when>
			<xsl:otherwise><xsl:value-of select="$trimmed-value"/></xsl:otherwise>
		</xsl:choose>
	</xsl:template>

		
	<!-- ************* Templates specific to PDF message formatting *************** -->
	<!-- attribute set for header block -->
	<xsl:attribute-set name="header-block">
		<xsl:attribute name="font-weight">bold</xsl:attribute>
		<xsl:attribute name="margin-top">1mm</xsl:attribute>
		<xsl:attribute name="margin-left">2mm</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for fixed message header block (denomination column) -->
	<xsl:attribute-set name="header-denom-column">
		<xsl:attribute name="font-size">9pt</xsl:attribute>
		<xsl:attribute name="font-weight">normal</xsl:attribute>
		<xsl:attribute name="border-top-color">black</xsl:attribute>
		<xsl:attribute name="border-top-width">0.5pt</xsl:attribute>
		<xsl:attribute name="border-top-style">solid</xsl:attribute>
		<xsl:attribute name="padding-bottom">3mm</xsl:attribute>
		<xsl:attribute name="margin-left">0mm</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for fixed message header block (value column) -->
	<xsl:attribute-set name="header-value-column">
		<xsl:attribute name="font-size">9pt</xsl:attribute>
		<xsl:attribute name="font-weight">bold</xsl:attribute>
		<xsl:attribute name="padding-bottom">3mm</xsl:attribute>
		<xsl:attribute name="margin-left">1.5mm</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for for the footer -->
	<xsl:attribute-set name="footer">
		<xsl:attribute name="font-size">9pt</xsl:attribute>
		<xsl:attribute name="font-weight">normal</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for information block left column (block title) -->
	<xsl:attribute-set name="main-left-column">
		<xsl:attribute name="font-size">9pt</xsl:attribute>
		<xsl:attribute name="font-weight">normal</xsl:attribute>
		<xsl:attribute name="margin-left">0mm</xsl:attribute>
		<xsl:attribute name="border-top-color">black</xsl:attribute>
		<xsl:attribute name="border-top-width">0.5pt</xsl:attribute>
		<xsl:attribute name="border-top-style">solid</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for information block right column (block contents of fixed size blocks) -->
	<xsl:attribute-set name="main-right-column">
		<xsl:attribute name="font-size">9pt</xsl:attribute>
		<xsl:attribute name="font-weight">bold</xsl:attribute>
		<xsl:attribute name="margin-left">1.5mm</xsl:attribute>
		<xsl:attribute name="border-left-color">black</xsl:attribute>
		<xsl:attribute name="border-left-width">0.5pt</xsl:attribute>
		<xsl:attribute name="border-left-style">solid</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for information block (first column, non-first rows of dynamic size blocks) -->
	<xsl:attribute-set name="table-left-column">
		<xsl:attribute name="font-size">9pt</xsl:attribute>
		<xsl:attribute name="font-weight">normal</xsl:attribute>
		<xsl:attribute name="margin-left">0mm</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for information block (description column for fixed size blocks) -->
	<xsl:attribute-set name="denom-column">
		<xsl:attribute name="font-size">9pt</xsl:attribute>
		<xsl:attribute name="font-weight">normal</xsl:attribute>
		<xsl:attribute name="margin-left">0mm</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for information block (table column title for dynamic size blocks) -->
	<xsl:attribute-set name="table-title-column">
		<xsl:attribute name="font-size">9pt</xsl:attribute>
		<xsl:attribute name="font-weight">normal</xsl:attribute>
		<xsl:attribute name="margin-left">1.5mm</xsl:attribute>
		<xsl:attribute name="border-top-color">black</xsl:attribute>
		<xsl:attribute name="border-top-width">0.5pt</xsl:attribute>
		<xsl:attribute name="border-top-style">solid</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for information block (description column for dynamic size blocks) -->
	<xsl:attribute-set name="table-denom-column">
		<xsl:attribute name="font-size">9pt</xsl:attribute>
		<xsl:attribute name="font-weight">normal</xsl:attribute>
		<xsl:attribute name="margin-left">1.5mm</xsl:attribute>
		<xsl:attribute name="border-left-color">black</xsl:attribute>
		<xsl:attribute name="border-left-width">0.5pt</xsl:attribute>
		<xsl:attribute name="border-left-style">solid</xsl:attribute>
	</xsl:attribute-set>
	
		<!-- attribute set for table header-->
	<xsl:attribute-set name="table-header-row">
		<xsl:attribute name="font-size">9pt</xsl:attribute>
		<xsl:attribute name="font-weight">bold</xsl:attribute>
		<xsl:attribute name="border-bottom-color">black</xsl:attribute>
		<xsl:attribute name="border-bottom-width">0.5pt</xsl:attribute>
		<xsl:attribute name="border-bottom-style">solid</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for table row -->
	<xsl:attribute-set name="table-row">
		<xsl:attribute name="font-size">9pt</xsl:attribute>
		<xsl:attribute name="font-weight">normal</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for table row -->
	<xsl:attribute-set name="table-main-row">
		<xsl:attribute name="font-size">9pt</xsl:attribute>
		<xsl:attribute name="font-weight">normal</xsl:attribute>
		<xsl:attribute name="border-bottom-color">black</xsl:attribute>
		<xsl:attribute name="border-bottom-width">0.5pt</xsl:attribute>
		<xsl:attribute name="border-bottom-style">solid</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for table row -->
	<xsl:attribute-set name="helper-text">
		<xsl:attribute name="font-size">9pt</xsl:attribute>
		<xsl:attribute name="font-weight">normal</xsl:attribute>
		<xsl:attribute name="padding-before">3pt</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for information block (value column) -->
	<xsl:attribute-set name="value-column">
		<xsl:attribute name="font-size">9pt</xsl:attribute>
		<xsl:attribute name="font-weight">bold</xsl:attribute>
		<xsl:attribute name="margin-left">1.5mm</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for information block (value-like description column for dynamic size blocks) -->
	<xsl:attribute-set name="first-value-column">
		<xsl:attribute name="font-size">10pt</xsl:attribute>
		<xsl:attribute name="font-weight">bold</xsl:attribute>
		<xsl:attribute name="margin-left">1.5mm</xsl:attribute>
		<xsl:attribute name="border-left-color">black</xsl:attribute>
		<xsl:attribute name="border-left-width">0.5pt</xsl:attribute>
		<xsl:attribute name="border-left-style">solid</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- standard template for a block containing only title and one string -->
	<xsl:template name="single-string-block">
		<xsl:param name="title"/>
		<xsl:param name="value"/>
		<fo:table table-layout="fixed" width="100%" space-after="{$space-after-blocks}" border-collapse="separate">
			<fo:table-column column-width="{$title-column-width}" column-number="1"/>
			<fo:table-column column-width="proportional-column-width(1)" column-number="2"/>
			<fo:table-body>
				<fo:table-row>
					<fo:table-cell column-number="1" xsl:use-attribute-sets="main-left-column">
						<fo:block><xsl:value-of select="$title"/></fo:block>
					</fo:table-cell>
					<fo:table-cell column-number="2" xsl:use-attribute-sets="main-right-column">
						<fo:block>
							<xsl:value-of select="."/>
						</fo:block>
					</fo:table-cell>
				</fo:table-row>
			</fo:table-body>
		</fo:table>
	</xsl:template>
	
	<!-- standard template for a block containing only title and one number -->
	<xsl:template name="single-number-block">
		<xsl:param name="title"/>
		<xsl:param name="value"/>
		<fo:table table-layout="fixed" width="100%" space-after="{$space-after-blocks}" border-collapse="separate">
			<fo:table-column column-width="{$title-column-width}" column-number="1"/>
			<fo:table-column column-width="{$description-column-width}" column-number="2"/>
			<fo:table-body>
				<fo:table-row>
					<fo:table-cell column-number="1" xsl:use-attribute-sets="main-left-column">
						<fo:block><xsl:value-of select="$title"/></fo:block>
					</fo:table-cell>
					<fo:table-cell column-number="2" xsl:use-attribute-sets="main-right-column">
						<fo:block>
							<xsl:value-of select="format-number($value, $integer-format)"/>
						</fo:block>
					</fo:table-cell>
				</fo:table-row>
			</fo:table-body>
		</fo:table>
	</xsl:template>
	
	<!-- standard template for a block containing only title and one amount (incl. cents) -->
	<xsl:template name="single-cents-block">
		<xsl:param name="title"/>
		<xsl:param name="value"/>
		<fo:table table-layout="fixed" width="100%" space-after="{$space-after-blocks}" border-collapse="separate">
			<fo:table-column column-width="{$title-column-width}" column-number="1"/>
			<fo:table-column column-width="{$description-column-width}" column-number="2"/>
			<fo:table-body>
				<fo:table-row>
					<fo:table-cell column-number="1" xsl:use-attribute-sets="main-left-column">
						<fo:block><xsl:value-of select="$title"/></fo:block>
					</fo:table-cell>
					<fo:table-cell column-number="2" xsl:use-attribute-sets="main-right-column">
						<fo:block>
							<xsl:value-of select="format-number($value, $cents-format)"/>
						</fo:block>
					</fo:table-cell>
				</fo:table-row>
			</fo:table-body>
		</fo:table>
	</xsl:template>
	
	<!-- standard template for a block containing only title and one date -->
	<xsl:template name="single-date-block">
		<xsl:param name="title"/>
		<xsl:param name="value"/>
		<fo:table table-layout="fixed" width="100%" space-after="{$space-after-blocks}" border-collapse="separate">
			<fo:table-column column-width="{$title-column-width}" column-number="1"/>
			<fo:table-column column-width="{$description-column-width}" column-number="2"/>
			<fo:table-body>
				<fo:table-row>
					<fo:table-cell column-number="1" xsl:use-attribute-sets="main-left-column">
						<fo:block><xsl:value-of select="$title"/></fo:block>
					</fo:table-cell>
					<fo:table-cell column-number="2" xsl:use-attribute-sets="main-right-column">
						<fo:block>
							<xsl:call-template name="format-date">
								<xsl:with-param name="value" select="$value"/>
							</xsl:call-template>
						</fo:block>
					</fo:table-cell>
				</fo:table-row>
			</fo:table-body>
		</fo:table>
	</xsl:template>
		
	<!-- Template for decoding the correction type-->
	<xsl:template name="correction-type">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="$trimmed-value = 'Current'"></xsl:when>
			<xsl:when test="$trimmed-value = 'CorrectionOld'">A</xsl:when>
			<xsl:when test="$trimmed-value = 'CorrectionNew'">N</xsl:when>
			<xsl:when test="$trimmed-value = 'Difference'">D</xsl:when>
			<xsl:otherwise><xsl:value-of select="$trimmed-value"/></xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	<!-- Template for outputting EMA values belonging to one taxatsource-->
	<xsl:template name="ema">
		<xsl:param name="taxatsource"/>
		<xsl:variable name="declarationcat" select="$taxatsource/*[local-name()='DeclarationCategory']"/>
		<xsl:if test="$declarationcat">
			<xsl:if test="$declarationcat/*[local-name()='Entry']">
				<xsl:call-template name="decode-ema">
					<xsl:with-param name="value" select="$declarationcat/*[local-name()='Entry']/*[local-name()='Reason']"/>
					<xsl:with-param name="type" select="concat('', 'Entry')"/>
				</xsl:call-template>
				<xsl:text> </xsl:text>			
			</xsl:if>
			<xsl:if test="$declarationcat/*[local-name()='Mutation']">
				<xsl:for-each select="$declarationcat/*[local-name()='Mutation']">
					<xsl:call-template name="decode-ema">
						<xsl:with-param name="value" select="./*[local-name()='Reason']"/>
						<xsl:with-param name="type" select="concat('', 'Mutation')"/>
					</xsl:call-template>
					<xsl:text> </xsl:text>
				</xsl:for-each>
			</xsl:if>
			<xsl:if test="$declarationcat/*[local-name()='Withdrawal']">
				<xsl:call-template name="decode-ema">
					<xsl:with-param name="value" select="$declarationcat/*[local-name()='Withdrawal']/*[local-name()='Reason']"/>
					<xsl:with-param name="type" select="concat('', 'Withdrawal')"/>
				</xsl:call-template>
				<xsl:text> </xsl:text>
			</xsl:if>
		</xsl:if>
	</xsl:template>
	
	<!-- Template for decoding the EMA value-->
	<xsl:template name="decode-ema">
		<xsl:param name="value"/>
		<xsl:param name="type"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="$trimmed-value = 'entryCompany'">E1</xsl:when>
			<xsl:when test="$trimmed-value = 'cantonChange' and $type='Entry'">E2</xsl:when>
			<xsl:when test="$trimmed-value = 'others' and $type='Entry'">E3</xsl:when>
			<xsl:when test="$trimmed-value = 'civilstate'">M1</xsl:when>
			<xsl:when test="$trimmed-value = 'partnerWork'">M2</xsl:when>
			<xsl:when test="$trimmed-value = 'sideline'">M3</xsl:when>
			<xsl:when test="$trimmed-value = 'partnerWorkIT'">M4</xsl:when>
			<xsl:when test="$trimmed-value = 'residence'">M5</xsl:when>
			<xsl:when test="$trimmed-value = 'childrenDeduction'">M6</xsl:when>
			<xsl:when test="$trimmed-value = 'churchTax'">M7</xsl:when>
			<xsl:when test="$trimmed-value = 'activityRate'">M8</xsl:when>
			<xsl:when test="$trimmed-value = 'others'  and $type='Mutation'">M9</xsl:when>
			<xsl:when test="$trimmed-value = 'withdrawalCompany'">A1</xsl:when>
			<xsl:when test="$trimmed-value = 'naturalization'">A2</xsl:when>
			<xsl:when test="$trimmed-value = 'settled-C'">A3</xsl:when>
			<xsl:when test="$trimmed-value = 'temporary'">A4</xsl:when>
			<xsl:when test="$trimmed-value = 'cantonChange' and $type='Withdrawal'">A5</xsl:when>
			<xsl:when test="$trimmed-value = 'others'  and $type='Withdrawal'">A6</xsl:when>
			<xsl:otherwise><xsl:value-of select="$trimmed-value"/></xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	<!-- Template for decoding tarif-->
	<xsl:template name="decode-taxatsource-category">
		<xsl:param name="value"/> <!-- TaxAtSourceCategory -->
			<xsl:if test="$value/*[local-name()='TaxAtSourceCode']">
				<xsl:value-of select="$value/*[local-name()='TaxAtSourceCode']"/>
			</xsl:if>
			<xsl:if test="$value/*[local-name()='CategoryPredefined']">
				<xsl:variable name="trimmed-value" select="normalize-space($value/*[local-name()='CategoryPredefined'])"/>
				<xsl:choose>
					<xsl:when test="$trimmed-value = 'specialAgreement'">AS</xsl:when>
					<xsl:when test="$trimmed-value = 'honoraryBoardOfDirectorsResidingAbroad'">CA</xsl:when>
					<xsl:when test="$trimmed-value = 'monetaryValuesServicesResidingAbroad'">PVD</xsl:when>
					<xsl:otherwise><xsl:value-of select="$trimmed-value"/></xsl:otherwise>
				</xsl:choose>
			</xsl:if>
			<xsl:if test="$value/*[local-name()='CategoryOpen']">AC</xsl:if>
	</xsl:template>
	
	<xsl:template match="*"/>

</xsl:stylesheet>