<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform" xmlns:fo="http://www.w3.org/1999/XSL/Format"
	xmlns:eCH-0007="http://www.ech.ch/xmlns/eCH-0007/4"
	xmlns:eCH-0010="http://www.ech.ch/xmlns/eCH-0010/4"
	xmlns:eCH-0011="http://www.ech.ch/xmlns/eCH-0011/5"
	xmlns:eCH-0044="http://www.ech.ch/xmlns/eCH-0044/2"
	xmlns:eCH-0046="http://www.ech.ch/xmlns/eCH-0046/1"
	xmlns:eCH-0058="http://www.ech.ch/xmlns/eCH-0058/3"
	xmlns:eahv-iv-common="http://www.eahv-iv.ch/xmlns/eahv-iv-common/2">
	
	<xsl:import href="sedex-id-1-0.xsl"/>
	<xsl:import href="ech-language-2-0_de.xsl"/>
	<xsl:import href="eahv-iv-common-language-2-0_de.xsl"/>
	
	<!-- Standard decimal format for swiss numbers. Empty numbers are shown as empty. -->
	<xsl:decimal-format grouping-separator="'" decimal-separator="." NaN=""/>
	<!-- Format string to use for integers -->
	<xsl:variable name="integer-format">###'###'###'###</xsl:variable>
	<!-- Format string to use for amounts including cents -->
	<xsl:variable name="cents-format">###'###'###'##0.00</xsl:variable>
	<!-- Format string to use for percentages. % sign is not added -->
	<xsl:variable name="percent-format">###.##</xsl:variable>
	
	<!-- Page margins -->
	<xsl:variable name="page-margins">13mm 13mm 8mm 13mm</xsl:variable>
	<xsl:variable name="first-header-height">0mm</xsl:variable>
	<xsl:variable name="header-height">11mm</xsl:variable>
	<xsl:variable name="footer-height">8mm</xsl:variable>
	<xsl:variable name="first-body-top-margin">0mm</xsl:variable>
	<xsl:variable name="body-top-margin">11mm</xsl:variable>
	<xsl:variable name="body-bottom-margin">12mm</xsl:variable>
	
	<!-- Some standard definitions for the layout of data blocks in PDF message layout -->
	<!-- White space between data blocks -->
	<xsl:variable name="space-after-blocks">6mm</xsl:variable>
	<!-- Font to use -->
	<xsl:variable name="font-family">Liberation Sans</xsl:variable>

	<!-- Width of the left header column on first page -->
	<xsl:variable name="header-left-column-width">92mm</xsl:variable>
	<!-- Width of the right header column on first page -->
	<xsl:variable name="header-right-column-width">proportional-column-width(1)</xsl:variable>
	<!-- Width of the title column (leftmost column of a data block) -->
	<xsl:variable name="title-column-width">38mm</xsl:variable>
	<!-- Width of the description column (second column of a data block) -->
	<xsl:variable name="description-column-width">50mm</xsl:variable>
	<!-- Width of the description column (second column of a data block), when in a table -->
	<xsl:variable name="description-column-table-width">proportional-column-width(1)</xsl:variable>
	<!-- Width of an interest column (showing interest rate) -->
	<xsl:variable name="interest-column-width">13mm</xsl:variable>
	<!-- Width of value column (showing amounts) -->
	<xsl:variable name="value-column-width">25mm</xsl:variable>
	<!-- Width of doctype column (showing amounts) -->
	<xsl:variable name="docpath-column-width">50mm</xsl:variable>
	<!-- Width of doctype column (showing amounts) -->
	<xsl:variable name="doctype-column-width">26mm</xsl:variable>
	
	<!-- Colordefinition for grey background -->
	<xsl:variable name="grey-background">#cdcdcd</xsl:variable>
	
	<xsl:variable name="lower-case">abcdefghijklmnopqrstuvwxyzäöüàéè</xsl:variable>
	<xsl:variable name="upper-case">ABCDEFGHIJKLMNOPQRSTUVWXYZÄÖÜÀÉÈ</xsl:variable>
	
	<!-- ************* Formatting templates *************** -->
	
	<!--
		Format a date for display in PDF.
		@param value the date to be formatted (as defined in xs:date)
	-->
	<xsl:template name="format-date">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="string-length($trimmed-value) > 0">
				<xsl:variable name="year" select="substring($trimmed-value, 1, 4)"/>
				<xsl:variable name="month" select="substring($trimmed-value, 6, 2)"/>
				<xsl:variable name="day" select="substring($trimmed-value, 9, 2)"/>
				<xsl:value-of select="$day"/>.<xsl:value-of select="$month"/>.<xsl:value-of select="$year"/>
			</xsl:when>
		</xsl:choose>
	</xsl:template>
	
	<!--
		Format a partially known date for display in PDF.
		@param value the date to be formatted (as defined in xs:data, xs:gYearMonth, xs:gYear)
	-->
	<xsl:template name="format-datePartiallyKnownType">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="string-length($trimmed-value) > 6">
				<xsl:call-template name="format-date">
					<xsl:with-param name="value" select="$value"/>
				</xsl:call-template>
			</xsl:when>
			<xsl:when test="string-length($trimmed-value) > 4">
				<xsl:variable name="year" select="substring($value, 1, 4)"/>
				<xsl:variable name="month" select="substring($value, 6, 2)"/>
				<xsl:value-of select="$month"/>.<xsl:value-of select="$year"/>
			</xsl:when>
			<xsl:when test="string-length($trimmed-value) > 0">
				<xsl:value-of select="$value"/>
			</xsl:when>
		</xsl:choose>
	</xsl:template>
	
	<!--
		Format a vn (=new AHV number), i.e. insert dots where appropriate.
		@param value vn to format
	-->
	<xsl:template name="format-vnType">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="string-length($trimmed-value) = 13">
				<xsl:value-of select="substring($trimmed-value, 1, 3)"/>.<xsl:value-of select="substring($trimmed-value, 4, 4)"/>.<xsl:value-of select="substring($trimmed-value, 8, 4)"/>.<xsl:value-of select="substring($trimmed-value, 12, 2)"/>
			</xsl:when>
			<xsl:otherwise>
				<xsl:value-of select="$trimmed-value"/>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	<!--
		Format an old AHV number, i.e. insert dots where appropriate.
		@param value vn to format
	-->
	<xsl:template name="format-old-ahv">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="string-length($trimmed-value) = 11">
				<xsl:value-of select="substring($trimmed-value, 1, 3)"/>.<xsl:value-of select="substring($trimmed-value, 4, 2)"/>.<xsl:value-of select="substring($trimmed-value, 6, 3)"/>.<xsl:value-of select="substring($trimmed-value, 9, 3)"/>
			</xsl:when>
			<xsl:otherwise>
				<xsl:value-of select="$trimmed-value"/>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	<!--
		Format a string, i.e. output the string
		@param value string to format
	-->
	<xsl:template name="format-string">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
				<xsl:value-of select="$trimmed-value"/>
	</xsl:template>
	
	<!--
		Format a string as a title, i.e. make sure the first character is upper case
		@param value string to format
	-->
	<xsl:template name="format-title">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:variable name="first-char" select="translate(substring($trimmed-value, 1, 1), $lower-case, $upper-case)"/>
		<xsl:value-of select="concat($first-char, substring($trimmed-value, 2))"/>
	</xsl:template>
	
	<!--
		Format a percentage, format a number and append a percent sign. If empty, the empty sign
		is output instead.
		@param value string to format
		@param empty-sign string to output if value is empty
	-->
	<xsl:template name="format-percent">
		<xsl:param name="value"/>
		<xsl:param name="empty-sign"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="string-length($trimmed-value) > 0">
				<xsl:value-of select="format-number($trimmed-value, $percent-format)"/>%
			</xsl:when>
			<xsl:otherwise><xsl:value-of select="$empty-sign"/></xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	
	<!-- ************* Decodings templates *************** -->
	
	<!--
		Decode an action to a human readable string.
		@param value the action to be decoded (as defined in ssk/eahv-iv)
	-->
	<xsl:template name="decode-action">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="$trimmed-value = '1'"><xsl:value-of select="$eahv-iv-common.action.1"/></xsl:when>
			<xsl:when test="$trimmed-value = '3'"><xsl:value-of select="$eahv-iv-common.action.3"/></xsl:when>
			<xsl:when test="$trimmed-value = '4'"><xsl:value-of select="$eahv-iv-common.action.4"/></xsl:when>
			<xsl:when test="$trimmed-value = '5'"><xsl:value-of select="$eahv-iv-common.action.5"/></xsl:when>
			<xsl:when test="$trimmed-value = '6'"><xsl:value-of select="$eahv-iv-common.action.6"/></xsl:when>
			<xsl:when test="$trimmed-value = '7'"><xsl:value-of select="$eahv-iv-common.action.7"/></xsl:when>
			<xsl:when test="$trimmed-value = '8'"><xsl:value-of select="$eahv-iv-common.action.8"/></xsl:when>
			<xsl:when test="$trimmed-value = '9'"><xsl:value-of select="$eahv-iv-common.action.9"/></xsl:when>
			<xsl:when test="$trimmed-value = '10'"><xsl:value-of select="$eahv-iv-common.action.10"/></xsl:when>
			<xsl:when test="$trimmed-value = '11'"><xsl:value-of select="$eahv-iv-common.action.11"/></xsl:when>
			<xsl:when test="$trimmed-value = '12'"><xsl:value-of select="$eahv-iv-common.action.12"/></xsl:when>
			<xsl:otherwise><xsl:value-of select="$trimmed-value"/></xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	<!--
		Decode a MIME type to a human readable string.
		@param value the type to be decoded
	-->
	<xsl:template name="decode-mime">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="$trimmed-value = 'image/tiff'">TIFF</xsl:when>
			<xsl:when test="$trimmed-value = 'application/pdf'">PDF</xsl:when>
			<xsl:when test="$trimmed-value = 'application/zip'">ZIP</xsl:when>
			<xsl:otherwise><xsl:value-of select="$trimmed-value"/></xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	<!--
		Decode a document type to a human readable string.
		@param value the type to be decoded
	-->
	<xsl:template name="decode-documentType">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="$trimmed-value = '000001'"><xsl:value-of select="$eahv-iv-common.documentType.000001"/></xsl:when>
			<xsl:when test="$trimmed-value = '000002'"><xsl:value-of select="$eahv-iv-common.documentType.000002"/></xsl:when>
			<xsl:when test="$trimmed-value = '000003'"><xsl:value-of select="$eahv-iv-common.documentType.000003"/></xsl:when>
			<xsl:when test="$trimmed-value = '000004'"><xsl:value-of select="$eahv-iv-common.documentType.000004"/></xsl:when>
			<xsl:when test="$trimmed-value = '000005'"><xsl:value-of select="$eahv-iv-common.documentType.000005"/></xsl:when>
			<xsl:when test="$trimmed-value = '000006'"><xsl:value-of select="$eahv-iv-common.documentType.000006"/></xsl:when>
			<xsl:when test="$trimmed-value = '000007'"><xsl:value-of select="$eahv-iv-common.documentType.000007"/></xsl:when>
			<xsl:when test="$trimmed-value = '000008'"><xsl:value-of select="$eahv-iv-common.documentType.000008"/></xsl:when>
			<xsl:when test="$trimmed-value = '000009'"><xsl:value-of select="$eahv-iv-common.documentType.000009"/></xsl:when>
			<xsl:when test="$trimmed-value = '000010'"><xsl:value-of select="$eahv-iv-common.documentType.000010"/></xsl:when>
			<xsl:when test="$trimmed-value = '000011'"><xsl:value-of select="$eahv-iv-common.documentType.000011"/></xsl:when>
			<xsl:when test="$trimmed-value = '000012'"><xsl:value-of select="$eahv-iv-common.documentType.000012"/></xsl:when>
			<xsl:when test="$trimmed-value = '000013'"><xsl:value-of select="$eahv-iv-common.documentType.000013"/></xsl:when>
			<xsl:when test="$trimmed-value = '000014'"><xsl:value-of select="$eahv-iv-common.documentType.000014"/></xsl:when>
			<xsl:when test="$trimmed-value = '000015'"><xsl:value-of select="$eahv-iv-common.documentType.000015"/></xsl:when>
			<xsl:when test="$trimmed-value = '000016'"><xsl:value-of select="$eahv-iv-common.documentType.000016"/></xsl:when>
			<xsl:when test="$trimmed-value = '000017'"><xsl:value-of select="$eahv-iv-common.documentType.000017"/></xsl:when>
			<xsl:when test="$trimmed-value = '000018'"><xsl:value-of select="$eahv-iv-common.documentType.000018"/></xsl:when>
			<xsl:when test="$trimmed-value = '000020'"><xsl:value-of select="$eahv-iv-common.documentType.000020"/></xsl:when>
			<xsl:when test="$trimmed-value = '000021'"><xsl:value-of select="$eahv-iv-common.documentType.000021"/></xsl:when>
			<xsl:when test="$trimmed-value = '000022'"><xsl:value-of select="$eahv-iv-common.documentType.000022"/></xsl:when>
			<xsl:when test="$trimmed-value = '000023'"><xsl:value-of select="$eahv-iv-common.documentType.000023"/></xsl:when>
			<xsl:when test="$trimmed-value = '000025'"><xsl:value-of select="$eahv-iv-common.documentType.000025"/></xsl:when>
			<xsl:when test="$trimmed-value = '000026'"><xsl:value-of select="$eahv-iv-common.documentType.000026"/></xsl:when>
			<xsl:when test="$trimmed-value = '000028'"><xsl:value-of select="$eahv-iv-common.documentType.000028"/></xsl:when>
			<xsl:when test="$trimmed-value = '999999'"><xsl:value-of select="$eahv-iv-common.documentType.999999"/></xsl:when>
			<xsl:otherwise><xsl:value-of select="$trimmed-value"/></xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	<!--
		Decode an personIdCategory to a human readable string.
		@param value the personIdCategory to be decoded (as defined in eCH-0044)
	-->
	<xsl:template name="decode-personIdCategory">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="$trimmed-value = 'CH.AHV'"><xsl:value-of select="$eCH-0044.personIdCategory.CH.AHV"/></xsl:when>
			<xsl:otherwise><xsl:value-of select="$trimmed-value"/></xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	<!--
		Decode a sex code to a human readable string. 
		@param value the sex to be decoded (as defined in eCH0044-sexType)
	-->
	<xsl:template name="decode-sexType">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="$trimmed-value = '1'"><xsl:value-of select="$eahv-iv-common.sex.1"/></xsl:when>
			<xsl:when test="$trimmed-value = '2'"><xsl:value-of select="$eahv-iv-common.sex.2"/></xsl:when>
			<xsl:otherwise><xsl:value-of select="$trimmed-value"/></xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	<!--
		Decode a religion code to a human readable string.
		@param value the religion to be decoded (as defined in eCH0011-religionType)
	-->
	<xsl:template name="decode-religionType">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="$trimmed-value = '111'"><xsl:value-of select="$eahv-iv-common.religion.111"/></xsl:when>
			<xsl:when test="$trimmed-value = '121'"><xsl:value-of select="$eahv-iv-common.religion.121"/></xsl:when>
			<xsl:when test="$trimmed-value = '122'"><xsl:value-of select="$eahv-iv-common.religion.122"/></xsl:when>
			<xsl:when test="$trimmed-value = '211'"><xsl:value-of select="$eahv-iv-common.religion.211"/></xsl:when>
			<xsl:when test="$trimmed-value = '711'"><xsl:value-of select="$eahv-iv-common.religion.711"/></xsl:when>
			<xsl:when test="$trimmed-value = '811'"><xsl:value-of select="$eahv-iv-common.religion.811"/></xsl:when>
			<xsl:when test="$trimmed-value = '000'"><xsl:value-of select="$eahv-iv-common.religion.000"/></xsl:when>
			<xsl:otherwise><xsl:value-of select="$trimmed-value"/></xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	<!--
		Decode a marital status code to a human readable string.
		@param value the marital status to be decoded (as defined in eCH0011-maritalStatusType)
	-->
	<xsl:template name="decode-maritalStatusType">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="$trimmed-value = '1'"><xsl:value-of select="$eahv-iv-common.maritalStatus.1"/></xsl:when>
			<xsl:when test="$trimmed-value = '2'"><xsl:value-of select="$eahv-iv-common.maritalStatus.2"/></xsl:when>
			<xsl:when test="$trimmed-value = '3'"><xsl:value-of select="$eahv-iv-common.maritalStatus.3"/></xsl:when>
			<xsl:when test="$trimmed-value = '4'"><xsl:value-of select="$eahv-iv-common.maritalStatus.4"/></xsl:when>
			<xsl:when test="$trimmed-value = '5'"><xsl:value-of select="$eahv-iv-common.maritalStatus.5"/></xsl:when>
			<xsl:when test="$trimmed-value = '6'"><xsl:value-of select="$eahv-iv-common.maritalStatus.6"/></xsl:when>
			<xsl:when test="$trimmed-value = '7'"><xsl:value-of select="$eahv-iv-common.maritalStatus.7"/></xsl:when>
			<xsl:otherwise><xsl:value-of select="$trimmed-value"/></xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	<!--
		Decode a type of real estate property to a human readable string. Unknown values are shown untranslated.
		@param value the marital status to be decoded (as defined in eahv-iv-common)
	-->
	<xsl:template name="decode-typeOfRealEstateProperty">
		<xsl:param name="value"/>
		<xsl:variable name="trimmed-value" select="normalize-space($value)"/>
		<xsl:choose>
			<xsl:when test="$trimmed-value = '0'"><xsl:value-of select="$eahv-iv-common.typeOfRealEstateProperty.0"/></xsl:when>
			<xsl:when test="$trimmed-value = '1'"><xsl:value-of select="$eahv-iv-common.typeOfRealEstateProperty.1"/></xsl:when>
			<xsl:when test="$trimmed-value = '2'"><xsl:value-of select="$eahv-iv-common.typeOfRealEstateProperty.2"/></xsl:when>
			<xsl:when test="$trimmed-value = '3'"><xsl:value-of select="$eahv-iv-common.typeOfRealEstateProperty.3"/></xsl:when>
			<xsl:when test="$trimmed-value = '4'"><xsl:value-of select="$eahv-iv-common.typeOfRealEstateProperty.4"/></xsl:when>
			<xsl:when test="$trimmed-value = '5'"><xsl:value-of select="$eahv-iv-common.typeOfRealEstateProperty.5"/></xsl:when>
			<xsl:when test="$trimmed-value = '6'"><xsl:value-of select="$eahv-iv-common.typeOfRealEstateProperty.6"/></xsl:when>
			<xsl:when test="$trimmed-value = '7'"><xsl:value-of select="$eahv-iv-common.typeOfRealEstateProperty.7"/></xsl:when>
			<xsl:when test="$trimmed-value = '8'"><xsl:value-of select="$eahv-iv-common.typeOfRealEstateProperty.8"/></xsl:when>
			<xsl:otherwise><xsl:value-of select="$trimmed-value"/></xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	<!-- ************* Templates specific to PDF message formatting *************** -->
	
	<!-- attribute set for fixed message header block (denomination column) -->
	<xsl:attribute-set name="header-denom-column">
		<xsl:attribute name="font-size">8pt</xsl:attribute>
		<xsl:attribute name="font-weight">normal</xsl:attribute>
		<xsl:attribute name="border-top-color">black</xsl:attribute>
		<xsl:attribute name="border-top-width">0.5pt</xsl:attribute>
		<xsl:attribute name="border-top-style">solid</xsl:attribute>
		<xsl:attribute name="padding-bottom">3mm</xsl:attribute>
		<xsl:attribute name="margin-left">0mm</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for fixed message header block (value column) -->
	<xsl:attribute-set name="header-value-column">
		<xsl:attribute name="font-size">8pt</xsl:attribute>
		<xsl:attribute name="font-weight">bold</xsl:attribute>
		<xsl:attribute name="padding-bottom">3mm</xsl:attribute>
		<xsl:attribute name="margin-left">1.5mm</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for for the footer -->
	<xsl:attribute-set name="footer">
		<xsl:attribute name="font-size">8pt</xsl:attribute>
		<xsl:attribute name="font-weight">normal</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for information block left column (block title) -->
	<xsl:attribute-set name="main-left-column">
		<xsl:attribute name="font-size">8pt</xsl:attribute>
		<xsl:attribute name="font-weight">normal</xsl:attribute>
		<xsl:attribute name="margin-left">0mm</xsl:attribute>
		<xsl:attribute name="border-top-color">black</xsl:attribute>
		<xsl:attribute name="border-top-width">0.5pt</xsl:attribute>
		<xsl:attribute name="border-top-style">solid</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for information block right column (block contents of fixed size blocks) -->
	<xsl:attribute-set name="main-right-column">
		<xsl:attribute name="font-size">8pt</xsl:attribute>
		<xsl:attribute name="font-weight">bold</xsl:attribute>
		<xsl:attribute name="margin-left">1.5mm</xsl:attribute>
		<xsl:attribute name="border-left-color">black</xsl:attribute>
		<xsl:attribute name="border-left-width">0.5pt</xsl:attribute>
		<xsl:attribute name="border-left-style">solid</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for information block (first column, non-first rows of dynamic size blocks) -->
	<xsl:attribute-set name="table-left-column">
		<xsl:attribute name="font-size">8pt</xsl:attribute>
		<xsl:attribute name="font-weight">normal</xsl:attribute>
		<xsl:attribute name="margin-left">0mm</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for information block (description column for fixed size blocks) -->
	<xsl:attribute-set name="denom-column">
		<xsl:attribute name="font-size">8pt</xsl:attribute>
		<xsl:attribute name="font-weight">normal</xsl:attribute>
		<xsl:attribute name="margin-left">0mm</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for information block (table column title for dynamic size blocks) -->
	<xsl:attribute-set name="table-title-column">
		<xsl:attribute name="font-size">8pt</xsl:attribute>
		<xsl:attribute name="font-weight">normal</xsl:attribute>
		<xsl:attribute name="margin-left">1.5mm</xsl:attribute>
		<xsl:attribute name="border-top-color">black</xsl:attribute>
		<xsl:attribute name="border-top-width">0.5pt</xsl:attribute>
		<xsl:attribute name="border-top-style">solid</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for information block (description column for dynamic size blocks) -->
	<xsl:attribute-set name="table-denom-column">
		<xsl:attribute name="font-size">8pt</xsl:attribute>
		<xsl:attribute name="font-weight">normal</xsl:attribute>
		<xsl:attribute name="margin-left">1.5mm</xsl:attribute>
		<xsl:attribute name="border-left-color">black</xsl:attribute>
		<xsl:attribute name="border-left-width">0.5pt</xsl:attribute>
		<xsl:attribute name="border-left-style">solid</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for information block (value column) -->
	<xsl:attribute-set name="value-column">
		<xsl:attribute name="font-size">8pt</xsl:attribute>
		<xsl:attribute name="font-weight">bold</xsl:attribute>
		<xsl:attribute name="margin-left">1.5mm</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- attribute set for information block (value-like description column for dynamic size blocks) -->
	<xsl:attribute-set name="first-value-column">
		<xsl:attribute name="font-size">8pt</xsl:attribute>
		<xsl:attribute name="font-weight">bold</xsl:attribute>
		<xsl:attribute name="margin-left">1.5mm</xsl:attribute>
		<xsl:attribute name="border-left-color">black</xsl:attribute>
		<xsl:attribute name="border-left-width">0.5pt</xsl:attribute>
		<xsl:attribute name="border-left-style">solid</xsl:attribute>
	</xsl:attribute-set>
	
	<!-- standard template for a block containing only title and one number -->
	<xsl:template name="single-number-block">
		<xsl:param name="title"/>
		<xsl:param name="value"/>
		<fo:table table-layout="fixed" width="100%" space-after="{$space-after-blocks}" border-collapse="separate">
			<fo:table-column column-width="{$title-column-width}" column-number="1"/>
			<fo:table-column column-width="{$description-column-width}" column-number="2"/>
			<fo:table-body>
				<fo:table-row>
					<fo:table-cell column-number="1" xsl:use-attribute-sets="main-left-column">
						<fo:block><xsl:value-of select="$title"/></fo:block>
					</fo:table-cell>
					<fo:table-cell column-number="2" xsl:use-attribute-sets="main-right-column">
						<fo:block>
							<xsl:value-of select="format-number($value, $integer-format)"/>
						</fo:block>
					</fo:table-cell>
				</fo:table-row>
			</fo:table-body>
		</fo:table>
	</xsl:template>
	
	<!-- standard template for a block containing only title and one amount (incl. cents) -->
	<xsl:template name="single-cents-block">
		<xsl:param name="title"/>
		<xsl:param name="value"/>
		<fo:table table-layout="fixed" width="100%" space-after="{$space-after-blocks}" border-collapse="separate">
			<fo:table-column column-width="{$title-column-width}" column-number="1"/>
			<fo:table-column column-width="{$description-column-width}" column-number="2"/>
			<fo:table-body>
				<fo:table-row>
					<fo:table-cell column-number="1" xsl:use-attribute-sets="main-left-column">
						<fo:block><xsl:value-of select="$title"/></fo:block>
					</fo:table-cell>
					<fo:table-cell column-number="2" xsl:use-attribute-sets="main-right-column">
						<fo:block>
							<xsl:value-of select="format-number($value, $cents-format)"/>
						</fo:block>
					</fo:table-cell>
				</fo:table-row>
			</fo:table-body>
		</fo:table>
	</xsl:template>
	
	<!-- standard template for a block containing only title and one date -->
	<xsl:template name="single-date-block">
		<xsl:param name="title"/>
		<xsl:param name="value"/>
		<fo:table table-layout="fixed" width="100%" space-after="{$space-after-blocks}" border-collapse="separate">
			<fo:table-column column-width="{$title-column-width}" column-number="1"/>
			<fo:table-column column-width="{$description-column-width}" column-number="2"/>
			<fo:table-body>
				<fo:table-row>
					<fo:table-cell column-number="1" xsl:use-attribute-sets="main-left-column">
						<fo:block><xsl:value-of select="$title"/></fo:block>
					</fo:table-cell>
					<fo:table-cell column-number="2" xsl:use-attribute-sets="main-right-column">
						<fo:block>
							<xsl:call-template name="format-date">
								<xsl:with-param name="value" select="$value"/>
							</xsl:call-template>
						</fo:block>
					</fo:table-cell>
				</fo:table-row>
			</fo:table-body>
		</fo:table>
	</xsl:template>
	
	<!-- helper template for outputting an eCH-0044:namedPersonIdType on one line-->
	<xsl:template name="format-personId">
		<xsl:param name="id-category"/>
		<xsl:param name="id"/>
		<xsl:choose>
			<xsl:when test="$id-category = 'CH.AHV'">
				<xsl:call-template name="format-old-ahv">
					<xsl:with-param name="value"><xsl:value-of select="$id"/></xsl:with-param>
				</xsl:call-template>
			</xsl:when>
			<xsl:otherwise>
				<xsl:call-template name="format-string">
					<xsl:with-param name="value"><xsl:value-of select="$id"/></xsl:with-param>
				</xsl:call-template>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	<!-- helper template for outputting eCH-0044:namedPersonIdType on one line-->
	<xsl:template match="eahv-iv-common:localPersonId | eahv-iv-common:otherPersonId | eahv-iv-common:euPersonId" mode="inline">
		<xsl:if test="position() != 1"><xsl:text>, </xsl:text></xsl:if>
		<xsl:call-template name="format-personId">
			<xsl:with-param name="id-category"><xsl:value-of select="eCH-0044:personIdCategory"/></xsl:with-param>
			<xsl:with-param name="id"><xsl:value-of select="eCH-0044:personId"/></xsl:with-param>
		</xsl:call-template>
		(<xsl:call-template name="decode-personIdCategory">
			<xsl:with-param name="value"><xsl:value-of select="eCH-0044:personIdCategory"/></xsl:with-param>
		</xsl:call-template>)
	</xsl:template>

	<!-- helper template for formatting eahv-iv-common.officialName -->
	<xsl:template match="eahv-iv-common:officialName">
		<fo:table-row>
			<fo:table-cell column-number="1" xsl:use-attribute-sets="denom-column">
				<fo:block><xsl:value-of select="$eahv-iv-common.officialName"/></fo:block>
			</fo:table-cell>
			<fo:table-cell column-number="2" xsl:use-attribute-sets="value-column">
				<fo:block>
					<xsl:value-of select="."/>
				</fo:block>
			</fo:table-cell>
		</fo:table-row>
	</xsl:template>
	
	<!-- helper template for formatting eahv-iv-common.firstName -->
	<xsl:template match="eahv-iv-common:firstName">
		<fo:table-row>
			<fo:table-cell column-number="1" xsl:use-attribute-sets="denom-column">
				<fo:block><xsl:value-of select="$eahv-iv-common.firstName"/></fo:block>
			</fo:table-cell>
			<fo:table-cell column-number="2" xsl:use-attribute-sets="value-column">
				<fo:block>
					<xsl:value-of select="."/>
				</fo:block>
			</fo:table-cell>
		</fo:table-row>
	</xsl:template>
	
	<!-- helper template for formatting eahv-iv-common.sex -->
	<xsl:template match="eahv-iv-common:sex">
		<fo:table-row>
			<fo:table-cell column-number="1" xsl:use-attribute-sets="denom-column">
				<fo:block><xsl:value-of select="$eahv-iv-common.sex"/></fo:block>
			</fo:table-cell>
			<fo:table-cell column-number="2" xsl:use-attribute-sets="value-column">
				<fo:block>
					<xsl:call-template name="decode-sexType">
						<xsl:with-param name="value"><xsl:value-of select="."/></xsl:with-param>
					</xsl:call-template>
				</fo:block>
			</fo:table-cell>
		</fo:table-row>
	</xsl:template>
	
	<!-- helper template for formatting eahv-iv-common.dateOfBirth -->
	<xsl:template match="eahv-iv-common:dateOfBirth">
		<fo:table-row>
			<fo:table-cell column-number="1" xsl:use-attribute-sets="denom-column">
				<fo:block><xsl:value-of select="$eahv-iv-common.dateOfBirth"/></fo:block>
			</fo:table-cell>
			<fo:table-cell column-number="2" xsl:use-attribute-sets="value-column">
				<fo:block>
					<xsl:call-template name="format-datePartiallyKnownType">
						<xsl:with-param name="value"><xsl:value-of select="."/></xsl:with-param>
					</xsl:call-template>
				</fo:block>
			</fo:table-cell>
		</fo:table-row>
	</xsl:template>
	
	<!-- helper template for formatting eahv-iv-common.vn -->
	<xsl:template match="eahv-iv-common:vn">
		<fo:table-row>
			<fo:table-cell column-number="1" xsl:use-attribute-sets="denom-column">
				<fo:block><xsl:value-of select="$eahv-iv-common.vn"/></fo:block>
			</fo:table-cell>
			<fo:table-cell column-number="2" xsl:use-attribute-sets="value-column">
				<fo:block>
					<xsl:call-template name="format-vnType">
						<xsl:with-param name="value"><xsl:value-of select="."/></xsl:with-param>
					</xsl:call-template>
				</fo:block>
			</fo:table-cell>
		</fo:table-row>
	</xsl:template>
	
	<!-- helper template for outputting an eCH-0044:namedPersonIdType on multiple lines -->
	<xsl:template match="eahv-iv-common:localPersonId | eahv-iv-common:otherPersonId | eahv-iv-common:euPersonId">
		<fo:table-row>
			<fo:table-cell column-number="1" xsl:use-attribute-sets="denom-column">
				<fo:block>
					<xsl:call-template name="decode-personIdCategory">
						<xsl:with-param name="value"><xsl:value-of select="eCH-0044:personIdCategory"/></xsl:with-param>
					</xsl:call-template>
				</fo:block>
			</fo:table-cell>
			<fo:table-cell column-number="2" xsl:use-attribute-sets="value-column">
				<fo:block>
					<xsl:call-template name="format-personId">
						<xsl:with-param name="id-category"><xsl:value-of select="eCH-0044:personIdCategory"/></xsl:with-param>
						<xsl:with-param name="id"><xsl:value-of select="eCH-0044:personId"/></xsl:with-param>
					</xsl:call-template>
				</fo:block>
			</fo:table-cell>
		</fo:table-row>
	</xsl:template>
	
	<!-- helper template for outputting an eCH-0097:namedOrganisationIdType on one line-->
	<xsl:template name="format-organisationId">
		<xsl:param name="id-category"/>
		<xsl:param name="id"/>
		<xsl:choose>
			<xsl:when test="$id-category = ''">
				<!-- format some known category -->
			</xsl:when>
			<xsl:otherwise>
				<xsl:call-template name="format-string">
					<xsl:with-param name="value"><xsl:value-of select="$id"/></xsl:with-param>
				</xsl:call-template>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>

	<!-- helper template for outputting an address tag (eCH-0010:addressInfoType) on one line-->
	<xsl:template match="eahv-iv-common:address">
		<fo:table-row>
			<fo:table-cell column-number="1" xsl:use-attribute-sets="denom-column">
				<fo:block><xsl:value-of select="$eahv-iv-common.address"/></fo:block>
			</fo:table-cell>
			<fo:table-cell column-number="2" xsl:use-attribute-sets="value-column">
				<fo:block>
					<xsl:apply-templates select="*" mode="inline"/>
				</fo:block>
			</fo:table-cell>
		</fo:table-row>
	</xsl:template>
	
	<!-- helper template for formatting eahv-iv-common.maritalStatus -->
	<xsl:template match="eahv-iv-common:maritalStatus">
		<fo:table-row>
			<fo:table-cell column-number="1" xsl:use-attribute-sets="denom-column">
				<fo:block><xsl:value-of select="$eahv-iv-common.maritalStatus"/></fo:block>
			</fo:table-cell>
			<fo:table-cell column-number="2" xsl:use-attribute-sets="value-column">
				<fo:block>
					<xsl:call-template name="decode-maritalStatusType">
						<xsl:with-param name="value" select="."/>
					</xsl:call-template>
				</fo:block>
			</fo:table-cell>
		</fo:table-row>
	</xsl:template>
	
	<!-- helper template for formatting eahv-iv-common.religion -->
	<xsl:template match="eahv-iv-common:religion">
		<fo:table-row>
			<fo:table-cell column-number="1" xsl:use-attribute-sets="denom-column">
				<fo:block><xsl:value-of select="$eahv-iv-common.religion"/></fo:block>
			</fo:table-cell>
			<fo:table-cell column-number="2" xsl:use-attribute-sets="value-column">
				<fo:block>
					<xsl:call-template name="decode-religionType">
						<xsl:with-param name="value"><xsl:value-of select="."/></xsl:with-param>
					</xsl:call-template>
				</fo:block>
			</fo:table-cell>
		</fo:table-row>
	</xsl:template>
	
	<!-- helper template for formatting eahv-iv-common.dateOfMaritalStatus -->
	<xsl:template match="eahv-iv-common:dateOfMaritalStatus">
		<fo:table-row>
			<fo:table-cell column-number="1" xsl:use-attribute-sets="denom-column">
				<fo:block><xsl:value-of select="$eahv-iv-common.dateOfMaritalStatus"/></fo:block>
			</fo:table-cell>
			<fo:table-cell column-number="2" xsl:use-attribute-sets="value-column">
				<fo:block>
					<xsl:call-template name="format-date">
						<xsl:with-param name="value" select="."/>
					</xsl:call-template>
				</fo:block>
			</fo:table-cell>
		</fo:table-row>
	</xsl:template>
	
	<!-- helper template for formatting eahv-iv-common.dateOfEntry -->
	<xsl:template match="eahv-iv-common:dateOfEntry">
		<fo:table-row>
			<fo:table-cell column-number="1" xsl:use-attribute-sets="denom-column">
				<fo:block><xsl:value-of select="$eahv-iv-common.dateOfEntry"/></fo:block>
			</fo:table-cell>
			<fo:table-cell column-number="2" xsl:use-attribute-sets="value-column">
				<fo:block>
					<xsl:call-template name="format-date">
						<xsl:with-param name="value" select="."/>
					</xsl:call-template>
				</fo:block>
			</fo:table-cell>
		</fo:table-row>
	</xsl:template>
	
	<!-- helper template for formatting eahv-iv-common.dateOfDeparture -->
	<xsl:template match="eahv-iv-common:dateOfDeparture">
		<fo:table-row>
			<fo:table-cell column-number="1" xsl:use-attribute-sets="denom-column">
				<fo:block><xsl:value-of select="$eahv-iv-common.dateOfDeparture"/></fo:block>
			</fo:table-cell>
			<fo:table-cell column-number="2" xsl:use-attribute-sets="value-column">
				<fo:block>
					<xsl:call-template name="format-date">
						<xsl:with-param name="value" select="."/>
					</xsl:call-template>
				</fo:block>
			</fo:table-cell>
		</fo:table-row>
	</xsl:template>
	
	<xsl:template match="*"/>

</xsl:stylesheet>